#!/usr/bin/env python3
import rclpy
from rclpy.node import Node
from rclpy.duration import Duration
from ds_dbw_msgs.msg import TurnSignalCmd, TurnSignal

class BlinkerNode(Node):
    def __init__(self):
        super().__init__('mkz_blinker')

        # Parameters
        self.topic = self.declare_parameter('topic', '/turn_signal_cmd').value
        # one of: 'left', 'right', 'hazard', 'off'
        self.mode  = self.declare_parameter('mode',  'right').value
        self.duration_sec = float(self.declare_parameter('duration_sec', 3.0).value)
        self.rate_hz = float(self.declare_parameter('rate_hz', 50.0).value)  # ~20ms period

        # Publisher
        self.pub = self.create_publisher(TurnSignalCmd, self.topic, 10)

        # Map strings -> (enum_name, fallback_int) to be robust across minor enum name changes
        self.map = {
            'off':    ('NONE',   0),
            'left':   ('LEFT',   1),
            'right':  ('RIGHT',  2),
            'hazard': ('HAZARD', 3),
        }

        enum_name, fallback = self.map.get(self.mode, ('RIGHT', 2))
        enum_value = getattr(TurnSignal, enum_name, fallback)

        self.msg_on = TurnSignalCmd()
        self.msg_on.cmd.value = int(enum_value)

        self.msg_off = TurnSignalCmd()
        self.msg_off.cmd.value = int(getattr(TurnSignal, 'NONE', 0))

        self.stop_time = self.get_clock().now() + Duration(seconds=self.duration_sec)
        self.timer = self.create_timer(1.0 / self.rate_hz, self._tick)

        self.get_logger().info(
            f'BlinkerNode publishing {enum_name} on {self.topic} '
            f'for {self.duration_sec:.2f}s @ {self.rate_hz:.1f} Hz'
        )

    def _tick(self):
        now = self.get_clock().now()
        if now < self.stop_time:
            self.pub.publish(self.msg_on)
        else:
            # publish OFF a few times to ensure latch-off, then exit
            self.pub.publish(self.msg_off)
            rclpy.shutdown()

def main():
    rclpy.init()
    node = BlinkerNode()
    rclpy.spin(node)

if __name__ == '__main__':
    main()
