#!/usr/bin/env python3
# alarm_node.py
import math
import time
from typing import Optional, Set, Tuple

import numpy as np

import rclpy
from rclpy.node import Node
from rclpy.qos import QoSProfile, QoSReliabilityPolicy, QoSHistoryPolicy

from sensor_msgs.msg import PointCloud2
from sensor_msgs_py import point_cloud2 as pc2

# Dataspeed / Ford DBW: use MiscCmd for turn signals on /vehicle/misc_cmd
from dbw_ford_msgs.msg import MiscCmd, TurnSignal, ParkingBrakeCmd


class OusterProximityAlarm(Node):
    """
    Detect motion within a small cylindrical ROI around the LiDAR by comparing
    voxel occupancy between consecutive frames. On motion, publish a turn-signal
    command via dbw_ford_msgs/MiscCmd for a fixed hold duration.

    Assumes the vehicle is stationary. If ego moves, you should compensate using odom/IMU.
    """

    # Logical mapping we use internally; matches common enum values
    ENUM = {'NONE': 0, 'LEFT': 1, 'RIGHT': 2, 'HAZARD': 3}

    def __init__(self) -> None:
        super().__init__('ouster_proximity_alarm')

        # ---------- Parameters ----------
        self.declare_parameter('points_topic', '/ouster/points')
        self.declare_parameter('misc_topic', '/vehicle/misc_cmd')  # where DBW listens
        self.declare_parameter('radius_feet', 5.0)
        self.declare_parameter('z_min', -1.0)          # meters, relative to sensor
        self.declare_parameter('z_max',  1.5)
        self.declare_parameter('voxel_size_m', 0.10)   # 10 cm voxels
        self.declare_parameter('diff_fraction_threshold', 0.30)
        self.declare_parameter('min_voxels', 50)
        self.declare_parameter('hold_seconds', 10.0)
        self.declare_parameter('cooldown_seconds', 2.0)
        self.declare_parameter('pub_rate_hz', 10.0)
        self.declare_parameter('turn_signal_mode', 'RIGHT')  # RIGHT or HAZARD

        # Read parameters
        self.points_topic = self.get_parameter('points_topic').get_parameter_value().string_value
        self.misc_topic   = self.get_parameter('misc_topic').get_parameter_value().string_value
        self.radius_m     = self.get_parameter('radius_feet').get_parameter_value().double_value * 0.3048
        self.z_min        = self.get_parameter('z_min').get_parameter_value().double_value
        self.z_max        = self.get_parameter('z_max').get_parameter_value().double_value
        self.voxel        = self.get_parameter('voxel_size_m').get_parameter_value().double_value
        self.diff_thresh  = self.get_parameter('diff_fraction_threshold').get_parameter_value().double_value
        self.min_voxels   = int(self.get_parameter('min_voxels').get_parameter_value().integer_value or 50)
        self.hold_seconds = self.get_parameter('hold_seconds').get_parameter_value().double_value
        self.cooldown_seconds = self.get_parameter('cooldown_seconds').get_parameter_value().double_value
        self.pub_rate_hz  = self.get_parameter('pub_rate_hz').get_parameter_value().double_value
        self.turn_signal_mode = self.get_parameter('turn_signal_mode').get_parameter_value().string_value.upper()
        if self.turn_signal_mode not in self.ENUM:
            self.get_logger().warn(f"turn_signal_mode '{self.turn_signal_mode}' not recognized; defaulting to RIGHT")
            self.turn_signal_mode = 'RIGHT'

        # ---------- Sub & Pub ----------
        qos = QoSProfile(depth=5)
        qos.reliability = QoSReliabilityPolicy.BEST_EFFORT
        qos.history = QoSHistoryPolicy.KEEP_LAST
        self.sub = self.create_subscription(PointCloud2, self.points_topic, self.cloud_cb, qos)

        self.pub = self.create_publisher(MiscCmd, self.misc_topic, 10)

        # ---------- State ----------
        self.prev_voxels: Optional[Set[Tuple[int, int, int]]] = None
        self.alarm_active_until: float = 0.0
        self.cooldown_until: float = 0.0
        self._pub_timer = None

        self.get_logger().info(
            f"ouster_alarm started. ROI radius={self.radius_m:.3f} m, z∈[{self.z_min},{self.z_max}] m, "
            f"voxel={self.voxel:.2f} m, diff_thresh={self.diff_thresh:.2f}, hold={self.hold_seconds:.1f}s, "
            f"mode={self.turn_signal_mode} → topic {self.misc_topic} | listening to {self.points_topic}"
        )

    # ---------------- Core logic ----------------
    def cloud_cb(self, msg: PointCloud2) -> None:
        voxels = self._voxels_from_cloud(msg)
        if voxels is None:
            return

        # First frame seeds the model
        if self.prev_voxels is None:
            self.prev_voxels = voxels
            return

        if len(voxels) < self.min_voxels:
            # Not enough density inside ROI — update and skip
            self.prev_voxels = voxels
            return

        # Fraction of new voxels in current frame relative to current occupancy
        inter = len(voxels.intersection(self.prev_voxels))
        frac_new = 1.0 - (inter / max(1, len(voxels)))
        motion = frac_new >= self.diff_thresh

        # Debug line (enable with --log-level ouster_proximity_alarm:=debug)
        self.get_logger().debug(
            f"voxels curr={len(voxels)} prev={len(self.prev_voxels)} inter={inter} "
            f"frac_new={frac_new:.3f} motion={motion}"
        )

        self.prev_voxels = voxels

        t_now = time.time()
        if motion and t_now >= self.cooldown_until and t_now >= self.alarm_active_until:
            self._activate_alarm()

    def _voxels_from_cloud(self, msg: PointCloud2) -> Optional[Set[Tuple[int, int, int]]]:
        try:
            pts_iter = pc2.read_points(msg, field_names=('x', 'y', 'z'), skip_nans=True)
        except Exception as e:
            self.get_logger().error(f'PointCloud2 parsing error: {e}')
            return None

        r2 = self.radius_m * self.radius_m
        v = self.voxel
        voxels: Set[Tuple[int, int, int]] = set()

        # You can add a stride here if needed for speed (e.g., keep every nth point)
        for x, y, z in pts_iter:
            if z < self.z_min or z > self.z_max:
                continue
            if (x * x + y * y) > r2:
                continue
            vx = int(math.floor(x / v))
            vy = int(math.floor(y / v))
            vz = int(math.floor(z / v))
            voxels.add((vx, vy, vz))
        return voxels

    def _activate_alarm(self) -> None:
        t_now = time.time()
        self.alarm_active_until = t_now + self.hold_seconds
        self.cooldown_until = self.alarm_active_until + self.cooldown_seconds

        if self._pub_timer is not None:
            self._pub_timer.cancel()

        period = 1.0 / max(1e-3, self.pub_rate_hz)
        self._pub_timer = self.create_timer(period, self._alarm_publish_step)
        self.get_logger().info(f"MOTION: Activating {self.turn_signal_mode} for {self.hold_seconds:.1f}s")

    def _alarm_publish_step(self) -> None:
        t_now = time.time()
        if t_now <= self.alarm_active_until:
            self._publish_turn_signal(self.turn_signal_mode)
        else:
            self._publish_turn_signal('NONE')
            if self._pub_timer is not None:
                self._pub_timer.cancel()
                self._pub_timer = None
            self.get_logger().info("Alarm done; turn signals OFF")

    def _publish_turn_signal(self, mode: str) -> None:
        """Publish dbw_ford_msgs/MiscCmd with TurnSignal value."""
        val = int(self.ENUM.get(mode.upper(), 0))
        msg = MiscCmd()
        # Prefer the TurnSignal enum if present, but raw values also work
        enum_val = getattr(TurnSignal, mode.upper(), val)
        msg.cmd.value = int(enum_val)
        if hasattr(msg, 'pbrk'):
            msg.pbrk.cmd = getattr(ParkingBrakeCmd, 'NONE', 0)
        self.pub.publish(msg)


def main() -> None:
    rclpy.init()
    try:
        node = OusterProximityAlarm()
    except Exception as e:
        print(f"Failed to start node: {e}")
        rclpy.shutdown()
        return
    try:
        rclpy.spin(node)
    except KeyboardInterrupt:
        pass
    finally:
        node.destroy_node()
        rclpy.shutdown()


if __name__ == '__main__':
    main()

